{ 
  open Parser
}

let alpha = ['a'-'z' 'A'-'Z']
let digit = ['0'-'9']
let ascii = [' '-'!' '#'-'[' ']'-'~']
let whitespace = [' ' '\t' '\r']
let escape_chars = ['\\' '"' 'n' 't' 'r']
let number = digit+ '.'? digit*
let str = (ascii | '\\' escape_chars)*
let id = ['a'-'z'] (alpha | digit | '_')*

rule token = parse
  whitespace    { token lexbuf }
  |  '#'        { comment lexbuf }
  | '\n'        { EOL }
  | '('         { LPAREN }
  | ')'         { RPAREN }
  | '['         { LSQUARE }
  | ']'         { RSQUARE }
  | '{'         { LBRACE }
  | '}'         { RBRACE }
  | ':'         { COLON }
  | ','         { COMMA }
  | '+'         { PLUS }
  | '-'         { MINUS }
  | '*'         { MULTIPLY }
  | '/'         { DIVIDE }
  | '%'         { MODULUS }
  | '<'         { LT }
  | '>'         { GT }
  | '='         { ASSIGN }
  | '!'         { NOT }
  | '^'         { CARET }
  | '@'         { APPEND }
  | "::"        { CONS }
  | "<="        { LTE }
  | ">="        { GTE }
  | "=="        { EQ }
  | "!="        { NEQ }
  | "and"       { AND }
  | "or"        { OR }
  | "not"       { NOT }
  | "lambda"    { LAMBDA }
  | "true"      { TRUE }
  | "false"     { FALSE }
  | "if"        { IF }
  | "elif"      { ELIF }
  | "else"      { ELSE }
  | "end"       { END }
  | "none"      { NONE }
  | eof         { EOF }
  | number as num         { NUM_LIT(float_of_string num) }
  | '"' (str as s) '"'    { STR_LIT(s) }
  | id as identifier      { ID(identifier) }
  | _ as char             { raise (Failure("SyntaxError: Invalid syntax -> " ^ Char.escaped char)) }

and comment = parse
    '\n'  { token lexbuf }
  | _     { comment lexbuf }
